{*******************************************************************************
                   Component: TGtroDBDateTimePicker
                          by Georges Trottier
                           http://www.gtro.com
                            17 November 2000

The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in compliance
with the License. You may obtain a copy of the License at
http://www.mozilla.org/NPL/NPL-1_1Final.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the License for
the specific language governing rights and limitations under the License.

The Initial Developer of the Code is GTRO Informatique (gtro@gtro.com)
Portions created by GTRO Informatique are Copyright (C) 2000 GTRO Informatique.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Name        : TGtroDBDateTimePicker, Version 1.0.0
Copyright   : Copyright(c)2000 GTRO Informatique
Description : TGtroBDateTimePicker is derived from TDateTimePicker from Delphi 5
            : TDateTimePicker Copyright(c)2000 Borland/Inprise.
            : Add capability to integrate with database
            : Whenever the content of the DateTimePicker changes, this content
            : is ready for posting in the selected field of the database.
            : Explicit posting is required to save the data in the database.
Date        : 17 November 2000
Author      : Georges Trottier
*******************************************************************************}

unit GtroDBDateTimePicker;

{$I GtroDBLib.INC}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, Db, DbCtrls;

type
  TCloseActions = (dtnChange, dtnCloseUp);

	TGtroDBDateTimePicker = class(TDateTimePicker)
  private
    FCloseAction: TCloseActions;
		FDataLink: TFieldDataLink; // we link only one field
    // delegation of TFieldDataLink properties to TGtroDBDateTimePicker
    function  GetDataField: string;
    function  GetDataSource: TDataSource;
    function  GetReadOnly: boolean;
    procedure SetDataField(const Value: string);
    procedure SetDataSource(const Value: TDataSource);
    procedure SetReadOnly(const Value: boolean);
		procedure DataChange(Sender: TObject);
		procedure UpdateData(Sender: TObject);
		procedure CNNotify(var Message: TWMNotify); message CN_NOTIFY;
  protected
		procedure Loaded; override;
  public
    constructor Create(Owner: TComponent); override;
    destructor Destroy; override;
  published
    property CloseAction: TCloseActions read FCloseAction write FCloseAction;
    property DataField: string read GetDataField write SetDataField;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
		property ReadOnly: boolean read GetReadOnly write SetReadOnly Default true;
  end;

procedure Register;

implementation

uses commctrl;

{ TGtroDBDateTimePicker - Private methods }

{*******************************************************************************
Function    : TGtroDBDateTimePicker.GetDataField
Parameters  : none
Result      : string
Visibility  : private
Description : read method for the DataField property
Revision    : November 17, 2000
Author      : Georges Trottier
*******************************************************************************}
function TGtroDBDateTimePicker.GetDataField: string;
begin
  Result:= FDataLink.FieldName;
end;

{*******************************************************************************
Function    : TGtroDBDateTimePicker.GetDataSource
Parameters  : none
Result      : Boolean
Visibility  : private
Description : read method for the DataSource property
Revision    : November 17, 2000
Author      : Georges Trottier
*******************************************************************************}
function TGtroDBDateTimePicker.GetDataSource: TDataSource;
begin
  Result:=  FDataLink.DataSource;
end;

{*******************************************************************************
Function    : TGtroDBDateTimePicker.GetReadOnly
Parameter   :
Result      : Boolean
Visibility  : private
Description : read method for the ReadOnly property
Revision    : November 17, 2000
Author      : Georges Trottier
*******************************************************************************}
function TGtroDBDateTimePicker.GetReadOnly: boolean;
begin
  Result:=  FDataLink.ReadOnly;
end;

{*******************************************************************************
Procedure   : TGtroDBDateTimePicker.SetDataField
Parameter   : const Value: string
Visibility  : private
Description : write method for ReadOnly property
Revision    : November 17, 2000
Author      : Georges Trottier
-------------------------------------------------------------------------------}
procedure TGtroDBDateTimePicker.SetDataField(const Value: string);
begin
  FDataLink.FieldName:= Value;
end;

{*******************************************************************************
procedure TGtroDBDateTimePicker.SetDataFiel
Parameter   : const Value: TDataSource
Visibility  : private
Description : Sets the DataSource; used to write the DataSource property
Revision    : November 17, 2000
Author      : Georges Trottier
*******************************************************************************}
procedure TGtroDBDateTimePicker.SetDataSource(const Value: TDataSource);
begin
  if not (FDataLink.DataSourceFixed and (csLoading in ComponentState)) then
    FDataLink.DataSource:= Value;
  if Value <> nil then
    Value.FreeNotification(self);
end;

{*******************************************************************************
Procedure   : TGtroDBDateTimePicker.SetReadOnly
Parameters  : const Value: boolean
Visibility  : private
Description : Sets the ReadOnly; used to write the ReadOnly property
Revision    : November 17, 2000
Author      : Georges Trottier
*******************************************************************************}
procedure TGtroDBDateTimePicker.SetReadOnly(const Value: boolean);
begin
	FDataLink.ReadOnly:= Value;
end;

{*******************************************************************************
Procedure   : TGtroDBDateTimePicker.DataChange
Parameters  : Sender: TObject
Visibility  : private
Description : Event handler for the OnDataChange event of the datalink
Revision    : November 17, 2000
Author      : Georges Trottier
*******************************************************************************}
procedure TGtroDBDateTimePicker.DataChange(Sender: TObject);
var
  DT: TDateTime;
begin
  if Assigned(FDataLink.Field) then DT:= FDataLink.Field.AsDateTime
  else DT:= Now;
  if Kind = dtkTime then Time:= DT
  else Date:= DT;
end;

{*******************************************************************************
Procedure   : TGtroDBDateTimePicker.UpdateData
Parameters  : Sender: TObject
Visibility  : private
Description : Event handler for the OnUpdateData event of the datalink
Revision    : November 17, 2000
Author      : Georges Trottier
*******************************************************************************}
procedure TGtroDBDateTimePicker.UpdateData(Sender: TObject);
begin
	try
		FDataLink.OnDataChange:= nil;
		FDataLink.DataSet.Edit;
		FDataLink.Field.AsDateTime:= Date;
	finally
		FDataLink.OnDataChange:= DataChange;
	end;
end;

{*******************************************************************************
Procedure   : TGtroDBDateTimePicker.CNNotify
Parameters  : var Message: TWMNotify (by reference)
Visibility  : private
Description : Intercepts CN_NOTIFY messages sent to the window and
            : reacts to DTN_CLOSEUP and DTN_DATETIMECHANGE.
            : This procedure overrides its ancestor's virtual procedure.
Revision    : November 17, 2000
Author      : Georges Trottier
*******************************************************************************}
procedure TGtroDBDateTimePicker.CNNotify(var Message: TWMNotify);
begin
	inherited; // do what DateTimePicker does before ...
	with Message, NMHdr^ do
  begin
		case code of
			DTN_DATETIMECHANGE:
        if CloseAction = dtnChange then UpdateData(self); // the system time has changed
      DTN_CLOSEUP:
        if CloseAction = dtnCloseUp then UpdateData(Self);
		end; //...case
	end; //...with
end;

{ Protected Method }

{*******************************************************************************
Procedure   : TGtroDBDateTimePicker.Loaded
Parameters  : none
Visibility  : protected
Description : Loaded calls it ancestor's method and allows the component to
            : show the content of the database during design.
            : This procedure overrides its ancestor's virtual procedure.
Revision    : November 17, 2000
Author      : Georges Trottier
*******************************************************************************}
procedure TGtroDBDateTimePicker.Loaded;
begin
  inherited;
  if csDesigning in ComponentState then DataChange(self);
end;

{ Public Methods }

{*******************************************************************************
Constructor : TGtroDBDateTimePicker.Create
Parameters  : none
Visibility  : public
Description : The constructor has the responsibility to generate a new instance
            : and proceed to the initialization of the parameters.
            : This constructor overrides its ancestor's virtual constructor.
Revision    : November 17, 2000
Author      : Georges Trottier
*******************************************************************************}
constructor TGtroDBDateTimePicker.Create(Owner: TComponent);
begin
  inherited;
  FDataLink:= TFieldDataLink.Create;
  FDataLink.Control:= self;
	FDataLink.OnDataChange:= DataChange;
	FDataLink.OnUpdateData:= UpdateData;
end;

{*******************************************************************************
Constructor : TGtroDBDateTimePicker.Destroy
Parameters  : none
Visibility  : public
Description : The destructor cleans up all garbage; it frees all the objects
            : that have been created in the constructor;
Revision    : November 17, 2000
Author      : Georges Trottier
*******************************************************************************}
destructor TGtroDBDateTimePicker.Destroy; // public
begin
	FDataLink.OnDataChange:= nil;
	FDataLink.OnUpdateData:= nil;
  FDataLink.Free;
  FDataLink:= nil;
  inherited;
end;

{ Global procedures }

{*******************************************************************************
Procedure   : Register
Parameter   : None
Visibility  : Global
Description : To Register in component Delphi component pallete
Revision    : November 17, 2000
Author      : Georges Trottier
Company     : GTRO Informatique, http://www.gtro.com, gtro @gtro.com
*******************************************************************************}
procedure Register;
begin
  RegisterComponents('GTRO', [TGtroDBDateTimePicker]);
end;

end.


