{*******************************************************************************
                       Component: TGtroDBTreeView
                          by Georges Trottier
                   Copyright(C) GTRO Informatique 2000
                           http://www.gtro.com
                            28 November 2000

The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in compliance
with the License. You may obtain a copy of the License at
http://www.mozilla.org/NPL/NPL-1_1Final.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the License for
the specific language governing rights and limitations under the License.

The Initial Developer of the Code is GTRO Informatique (gtro@gtro.com)
Portions created by GTRO Informatique are Copyright (C) 2000 GTRO Informatique.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Name        : TGtroDBDBTreeView, Version 1.0.0
Copyright   : Copyright(c)2000 GTRO Informatique
Description : TGtroDBTreeView is a class derived from TTreeView from Delphi 5
            : It integrates a TreeView component with a dataset and allows one
            : to manage hierarchical data. 
Date        : 28 November 2000
Author      : Georges Trottier
Note        : Needed function RecordToNode().Boolean
*******************************************************************************}

unit GTroDBTreeView;

{$I GtroDBLib.INC}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, commctrl, db, dbCtrls, dbtables;

const
  IMG_HOME = 0;
	IMG_FOLDER_CLOSED = 1;
  IMG_FILE_CLOSED = 2;
  IMG_FOLDER_OPEN = 3;
  IMG_FILE_OPEN = 4;

ResourceString
  streDelete = 'Delete this record?';
  strfDelete = 'Supprimet ce record?';
  streDependents = 'This node has dependents' + #13#10 + 'It cannot be deleted';
  strfDependents = 'Ce noeud a des dpendants' + #13#10 +
    'Il ne peut pas tre supprim';

type
	eNodeType = (ntRoot, ntFolder, ntFile); // types of nodes
  TNavConnectEvent = procedure (Sender: TObject; Connected: Boolean) of object;

{*******************************************************************************
                         Declaration of the TItem class
*******************************************************************************}
  TItem = class
  private
  	RecordID, ParentID: Integer;
    Text: string;
    FParentItem: TItem;
    Flag: Boolean;
  public
    constructor Create(F: Boolean; R, P: Integer; T: string);
    property ParentItem: TItem read FParentItem write FParentItem;
  end;

{*******************************************************************************
                       Declaration of the TListItems class
*******************************************************************************}
  TListOfItems = class(TList)
  	destructor Destroy; override;
    function FindItem(X: Integer): TItem;
  end;

{*******************************************************************************
                         Declaration of the datalink class
*******************************************************************************}
  TTreeViewDataLink = class(TDataLink)
  private
    FTreeView: TCustomTreeView;
    FFieldNames: array [0..3] of string; // four fields
    FFields: array [0..3] of TField;
    FOnActiveChange: TNotifyEvent;
    function GetFields(i: Integer): TField;
    function GetFieldNames(i: Integer): string;
    procedure SetFields(i: Integer; Value: TField);
    procedure SetFieldNames(I: Integer; const Value: string);
    procedure UpdateField(i: Integer);
  protected
    procedure ActiveChanged; override;
  public
    constructor Create(ATreeView: TCustomTreeView);
    property FieldNames[i: Integer]: string
      read GetFieldNames write SetFieldNames;
    property TreeView: TCustomTreeView read FTreeView write FTreeView;
    property Fields[i: Integer]: TField read GetFields;
    property OnActiveChange: TNotifyEvent read FOnActiveChange write FOnActiveChange;
 end;

{*******************************************************************************
                          Declaration of the component
*******************************************************************************}
  TGtroDBTreeView = class(TCustomTreeView)
  private
    FDataLink: TTreeViewDataLink;
    FRootCaption: string;
    FIsContainer: boolean;
    FCanDelete: Boolean;
  	FListOfItems: TListOfItems;
    FMessagesOnDelete: Boolean;
    FExtNavigator: TDBNavigator;
    FNavigator: TDBNavigator;
  	FNextRecordID: Integer;
    FOldOnClick: ENavClick;
    FOnChange: TTVChangedEvent;
    FOnClick: TNotifyEvent;
    FOnEdited: TTVEditedEvent;
    FOnEnter: TNotifyEvent;
    FOnExit: TNotifyEvent;
    FNodeType: ENodeType;
    RemButtons: TButtonSet;
    FImageList: TImageList;
    procedure AddRootNode;
    function  AddANode(Item: TItem): TTreeNode;
    procedure ActiveChange(Sender: TObject);
    procedure ConnectNavigator(Navigator: TDBNavigator);
    procedure DisconnectNavigator;
    procedure Edited(Sender: TObject; Node: TTreeNode; var S: string);
    function  FindNode(i: Integer): TTreeNode;
    function  IsNodeAllowed(ParentNode: TTreeNode): Boolean;
    function  GetDataSource: TDataSource;
    function  GetKeyFieldName: string;
    function  GetIsFolderFieldName: string;
		function  GetNodeType(ParentNode: TTreeNode): eNodeType;
    function  GetParentFieldName: string;
    function  GetNextRecordID: Integer;
    function  GetLabelFieldName: string;
    procedure LoadFromTable;
    procedure NavigatorClick(Sender: TObject; Button: TNavigateBtn);
    procedure SetDataSource(Value: TDataSource);
    procedure SetKeyFieldName(const Value: string);
    procedure SetIsFolderFieldName(const Value: string);
    procedure SetParentFieldName(const Value: string);
    procedure SetLabelFieldName(const Value: string);
    procedure TVChange(Sender: TObject; Node: TTreeNode);
    procedure TVClick(Sender: TObject);
    procedure TVDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure TVDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure TVEnter(Sender: TObject);
    procedure TVExit(Sender: TObject);
  protected
    procedure Edit(const Item: TTVItem); override;
    procedure Loaded; override;
    property NextRecordID: Integer read GetNextRecordID;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure AddNewNode(NodeType: ENodeType);
    function NodeToRecord(Node: TTreeNode): Boolean;
    function SelectNode(RecID: Integer): Boolean;
    procedure DeleteSelectedNode;
    procedure Focus;
    property IsContainer: boolean read FIsContainer;
    property NodeType: ENodeType read FNodeType write FNodeType;
  published
   property KeyFieldName: string
      read GetKeyFieldName write SetKeyFieldName;
    property ParentFieldName: string
      read GetParentFieldName write SetParentFieldName;
    property IsFolderFieldName: string
      read GetIsFolderFieldName write SetIsFolderFieldName;
    property LabelFieldName: string
      read GetLabelFieldName write SetLabelFieldName;
  	property DataSource: TDataSource read GetDataSource write SetDataSource;
    property MessagesOnDelete: Boolean
      read FMessagesOnDelete write FMessagesOnDelete;
    property Navigator: TDBNavigator read FExtNavigator write FExtNavigator;
    property RootCaption: string read FRootCaption write FRootCaption;
    property CanDelete: Boolean read FCanDelete;
    property Align;
    property Color;
    property Ctl3D;
    property DragCursor;
    property Enabled;
    property Font;
    property HideSelection;
    property Images;
    property Indent;
    property ParentColor default False;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly;
    property RightClickSelect;
    property ShowButtons;
    property ShowHint;
    property ShowLines;
    property ShowRoot;
    property StateImages;
    property TabOrder;
    property TabStop default True;
    property Visible;
    property OnChange;
    property OnChanging;
    property OnClick;
    property OnCollapsed;
    property OnCollapsing;
    property OnDblClick;
    property OnDeletion;
    property OnEdited;
    property OnEditing;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnExpanding;
    property OnExpanded;
    property OnGetImageIndex;
    property OnGetSelectedIndex;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    {$IFDEF GTRO_D4}
      property Anchors;
      property AutoExpand;
      property BiDiMode;
      property BorderStyle;
      property BorderWidth;
      property ChangeDelay;
      property Constraints;
      property DragKind;
      property HotTrack;
      property ParentBiDiMode;
      property RowSelect;
      property ToolTips;
      property OnCustomDraw;
      property OnCustomDrawItem;
      property OnEndDock;
      property OnStartDock;
    {$ENDIF}
    {$IFDEF GTRO_D5}
      property OnAdvancedCustomDraw;
      property OnAdvancedCustomDrawItem;
      property OnContextPopup;
    {$ENDIF}
    // property OnStartDrag;
    { Items must be published after OnGetImageIndex and OnGetSelectedIndex }
    // property Items;
  end;

procedure Register;

implementation

uses ImgList, ShellApi;

{*******************************************************************************
Function    : GetLangFlag
Visibility  : global
Description : Established what the user locale is. It returns true if the
            : language is french, false otherwise.
Revision    : 30 November 2000
*******************************************************************************}
function GetLangFlag: Boolean;
var
	Locale: LCID;
	LocaleName: array [0..128] of char;
begin
	Result:= False;
	Locale:= LCID(GetUserDefaultLCID);
	if GetLocaleInfo(Locale, LOCALE_SLANGUAGE, LocaleName, 128) = 0 then
		ShowMessage(SysErrorMessage(GetLastError));
	GetLocaleInfo(Locale, LOCALE_SABBREVLANGNAME, LocaleName, 128);
	if Copy(LocaleName, 0, 2) = 'fr' then Result:= True
end;

{*******************************************************************************
Function    : MyCustomSortProc
Visibility  : global
Description : Sorts the tree alphabetically but puts the files before the folders.
Revision    : 30 November 2000
*******************************************************************************}
function MyCustomSortProc(Node1, Node2: TTreeNode; Data: Integer): Integer; stdcall;

  function IsAFile(Node: TTreeNode): Boolean;
  begin
  	Result:= Node.ImageIndex = IMG_FILE_CLOSED;
  end;

  function IsAFolder(Node: TTreeNode): Boolean;
  begin
  	Result:= 	(Node.ImageIndex = IMG_FOLDER_CLOSED) or
    					(Node.ImageIndex = IMG_HOME);
  end;

begin
	if IsAFile(Node1) and IsAFolder(Node2) then
  begin // Files before folders
  	Result:= -1;
    Exit;
  end; // ...if
  if IsAFolder(Node1) and IsAFile(Node2) then
  begin
  	Result:= 1;
    Exit;
  end; // ...if
  Result:= AnsiStrComp(PChar(Node1.Text), PChar(Node2.Text));
end;

{*******************************************************************************
                             TListOfItems
*******************************************************************************}

{*******************************************************************************
Destructor  : TListOfItems.Destroy
Visibility  : public
Description : read method for the DataField property
Revision    : November 15, 2000
*******************************************************************************}
destructor TListOfItems.Destroy;
var
	i: Integer;
begin
	for i:= 0 to Count - 1 do TItem(Items[i]).Free;
end; // Destroy

{*******************************************************************************
Function    : TListOfItems.FindItem
Visibility  : public
Description : Find the item with the RecordID of X and returns a reference on
            : the Item
Revision    : November 15, 2000
*******************************************************************************}
function TListOfItems.FindItem(X: Integer): TItem;
var
	i: Integer;
begin
	Result:= nil;
	for i:= 0 to Count - 1 do
  begin
  	if TItem(Items[i]).RecordID = X then
    begin
    	Result:= TItem(Items[i]);
      Exit;
    end; // ...if
  end; // ...for
end; // FindItem


{*******************************************************************************
                                  TItems
*******************************************************************************}

{*******************************************************************************
Constructor : TItem.Create
Visibility  : public
Description : Creates an item in the List of Items
Revision    : November 15, 2000
*******************************************************************************}
constructor TItem.Create(F: Boolean; R, P: Integer; T: string);
begin
	inherited Create;
	Flag:= F;
	RecordID:= R;
  ParentID:= P;
  Text:= T;
  FParentItem:= nil;
end;

{*******************************************************************************
                            TTreeViewDatalink
*******************************************************************************}

constructor TTreeViewDataLink.Create(ATreeView: TCustomTreeView);
{*******************************************************************************
Constructor : TTreeViewDataLink.Create
Visibility  : public
Description : Creates a TTreeViewDataLink
Revision    : November 18, 2000
*******************************************************************************}
begin
  inherited Create;
  FTreeView:= ATreeView;
  {$IFDEF GTRO_D5} VisualControl:= True {$ENDIF};
end;

procedure TTreeViewDataLink.SetFields(i: Integer; Value: TField);
{*******************************************************************************
Procedure   : TTreeViewDataLink.SetFields
Visibility  : private
Description : write method for the property Fields
Revision    : November 18, 2000
*******************************************************************************}
begin
  if FFields[i] <> Value then FFields[i]:= Value;
end;

function TTreeViewDataLink.GetFields(i: Integer): TField;
{*******************************************************************************
Function    : TTreeViewDataLink.GetFields
Visibility  : private
Description : read method for the property Fields
Revision    : November 18, 2000
*******************************************************************************}
begin
  Result:= FFields[i]
end;

procedure TTreeViewDataLink.SetFieldNames(I: Integer; const Value: string);
{*******************************************************************************
Procedure   : TTreeViewDataLink.SetFieldNames
Visibility  : private
Description : write method for the property FieldNames
Revision    : November 18, 2000
*******************************************************************************}
begin
  FFieldNames[i]:= Value;
  UpdateField(i);
end;

{*******************************************************************************
Function    : TTreeViewDataLink.GetFieldNames
Visibility  : private
Description : read method for the property FieldNames
Revision    : November 18, 2000
*******************************************************************************}
function TTreeViewDataLink.GetFieldNames(i: Integer): string;
begin
  Result:= FFieldNames[i];
end;

{*******************************************************************************
Function    : TTreeViewDataLink.UpdateField
Visibility  : private
Description : Given that the Datalink is active and the FieldNames are defined,
            : it initializes the Fields property.
            : Called by ActiveChange
Revision    : November 18, 2000
*******************************************************************************}
procedure TTreeViewDataLink.UpdateField(i: Integer);
begin
  if Active and (FFieldNames[i] <> '') then
  begin
    if Assigned(FTreeView) then
      SetFields(i, GetFieldProperty(DataSource.DataSet, FTreeView, FFieldNames[i]))
    else
      SetFields(i, DataSource.DataSet.FieldByName(FFieldNames[i]));
  end
  else SetFields(i, nil);
end;

{*******************************************************************************
Procedure   : TTreeViewDataLink.ActiveChanged; override
Visibility  : Protected
Description : Activated by the OnActiveChange Event of the Datalink
Revision    : 30 Novembre 2000
*******************************************************************************}
procedure TTreeViewDataLink.ActiveChanged;
var
  i: Integer;
begin
  for i:= 0 to 3 do UpdateField(i);
  if Assigned(FOnActiveChange) then FOnActiveChange(Self);
end;

{*******************************************************************************
                     TGtroDBTreeView - Private Methods
*******************************************************************************}

procedure TGtroDBTreeView.AddRootNode;
{*******************************************************************************
Procedure   : TGtroDBTreeView.AddRootNode
Visibility  : Private
Description : Add the root node to the TreeView
            : Called by TGtroDBTreeView.LoadFromTable
Revision    : 30 Novembre 2000
*******************************************************************************}
begin
	if Items.Count = 0 then
  begin
  	with Items.AddFirst(nil, FRootCaption) do
    begin
    	Selected:= True;
    	ImageIndex:= IMG_HOME;
      SelectedIndex:= IMG_HOME;
      Data:= TObject(0);
    end; // ...with
  end; // ...if
end;

function TGtroDBTreeView.AddANode(Item: TItem): TTreeNode;
{*******************************************************************************
Function    : TGtroDBTreeView.AddANode
Visibility  : private
Description : Add a node to the TreeView during LoadFromTable
            : Called by LoadFromTable
Revision    : November 15, 2000
*******************************************************************************}
var
	ParentNode: TTreeNode;
  ParentItem: TItem;
begin
  Result:= FindNode(Item.RecordID);
  if Result = nil then
  begin
  	ParentItem:= Item.ParentItem;
    if ParentItem <> nil then ParentNode:= AddANode(ParentItem)
    else ParentNode:= Items.GetFirstNode;
  	Result:= Items.AddChildObject(ParentNode, Item.Text, TObject(Item.RecordID));
    if Item.Flag then
    begin
      Result.ImageIndex:= IMG_FOLDER_CLOSED;
      Result.SelectedIndex:= IMG_FOLDER_OPEN;
      Result.MakeVisible;
    end // ...if
    else
    begin
      Result.ImageIndex:= IMG_FILE_CLOSED;
      Result.SelectedIndex:= IMG_FILE_OPEN;
    end; // ...else
  end; // ...if
end;

procedure TGtroDBTreeView.ActiveChange(Sender: TObject);
{*******************************************************************************
Procedure   : TGtroDBTreeView.GoLoad
Visibility  : private
Description : Initiates loading of the TreeView from the database.
            : Called by TGtroDBTreeView.Create (assigned as event handler for
            : the OnActiveChange event of the Datalink)
Revision    : 4 December 2000
*******************************************************************************}
begin
  if (FDatalink.Active) and not (csDesigning in ComponentState) then
    LoadFromTable;
end;

procedure TGtroDBTreeView.Edited(Sender: TObject; Node: TTreeNode;
{*******************************************************************************
Procedure   : TGtroDBTreeView.Edited
Visibility  : private
Description : Posts the new title in the database.
            : Called by TGtroDBTreeView.Create (assigned as event handler for
            : the OnEdited event of the ancestor TreeView)
Revision    : 4 December 2000
*******************************************************************************}
var
  S: string);
begin
  with FDataLink.DataSource.DataSet do
  begin
  	Edit;
  	FDatalink.Fields[3].Value:= S;
  	Post;
  end; // ...with
  if Assigned(FOnEdited) then FOnEdited(Sender, Node, S);
end;

function TGtroDBTreeView.FindNode(i: Integer): TTreeNode;
{*******************************************************************************
Function    : TGtroDBTreeView.FindNode
Visibility  : private
Description : Add a node to the TreeView during LoadFromTable
            : Called by AddANode
Revision    : November 15, 2000
*******************************************************************************}
var
	Node: TTreeNode;
begin
	Result:= nil; // pas trouv
  Node:= Items.GetFirstNode;
  while Node <> nil do
  begin
  	if Integer(Node.Data) = i then
    begin
    	Result:= Node;
      Exit;
    end; // ...if
    Node:= Node.GetNext;
    Application.ProcessMessages;
  end; // ...while
end;

function TGtroDBTreeView.IsNodeAllowed(ParentNode: TTreeNode): Boolean;
{*******************************************************************************
Function    : TGtroDBTreeView.IsNodeAllowed
Visibility  : private
Description : Check if the node is a container of nodes
            : Called by TGtroDBTreeView.AddNewNode,
            : TGtroDBTreeView.TreeViewChange, TGtroDBTreeView.TVDragOver
Revision    : November 15, 2000
*******************************************************************************}
begin
	case GetNodeType(ParentNode) of
  	ntRoot	: Result:= True;
  	ntFolder: Result:= True;
  	ntFile	: Result:= False;
  	else
  		Result:= False;
  end;
end;

function TGtroDBTreeView.GetDataSource: TDataSource;
{*******************************************************************************
Function    : TGtroDBTreeView.GetDataSource
Visibility  : private
Description : Read method for the property DataSource
Revision    : November 15, 2000
*******************************************************************************}
begin
  Result := FDataLink.DataSource;
end;

function TGtroDBTreeView.GetKeyFieldName: string;
{*******************************************************************************
Function    : TGtroDBTreeView.GetKeyFieldName
Visibility  : private
Description : Read method for the property KeyFieldName
Revision    : November 15, 2000
*******************************************************************************}
begin
  Result:= FDatalink.FieldNames[0];
end;

function TGtroDBTreeView.GetIsFolderFieldName: string;
{*******************************************************************************
Function    : TGtroDBTreeView.GetIsFolderFieldName
Visibility  : private
Description : Read method for the property IsFolderFieldName
Revision    : November 15, 2000
*******************************************************************************}
begin
  Result:= FDatalink.FieldNames[2];
end;

function TGtroDBTreeView.GetNodeType(ParentNode: TTreeNode): eNodeType;
{*******************************************************************************
Function    : TGtroDBTreeView.GetNodeType
Visibility  : private
Description : Returns the type of a node
            : Called by TGtroDBTreeView.IsNodeAllowed
Revision    : November 15, 2000
*******************************************************************************}
begin
	case ParentNode.ImageIndex of
  	IMG_HOME					: Result:= ntRoot;
    IMG_FOLDER_CLOSED,
    IMG_FOLDER_OPEN	: Result:= ntFolder;
    IMG_FILE_CLOSED,
    IMG_FILE_OPEN		: Result:= ntFile;
  else Result:= ntFile;
	end; // ...case
end;

function TGtroDBTreeView.GetParentFieldName: string;
{*******************************************************************************
Function    : TGtroDBTreeView.GetParentFieldName
Visibility  : private
Description : Read method for the property ParentFieldName
Revision    : November 15, 2000
*******************************************************************************}
begin
  Result:= FDatalink.FieldNames[1];
end;

function TGtroDBTreeView.GetNextRecordID: Integer;
{*******************************************************************************
Function    : TGtroDBTreeView.GetNextRecordID
Visibility  : private
Description : Read method for the property NextRecordID
            : Returns the next available key for the KeyField of the table
            : Called by TGtroDBTreeView.AddNewNode
Revision    : November 15, 2000
*******************************************************************************}
var
	BookMark: TBookMark;
begin
  Result:= 1;
  if FDatalink.DataSource.DataSet.RecordCount <> 0 then
  begin
    BookMark:= FDatalink.DataSource.DataSet.GetBookMark;
    try
      FDataLink.DataSource.DataSet.Last;
      FNextRecordID:= FDatalink.Fields[0].Value + 1;
      Result:= FNextRecordID;
    finally
      FDatalink.DataSource.DataSet.GoToBookMark(BookMark);
      FDatalink.DataSource.DataSet.FreeBookMark(BookMark);
    end; // try...finally
  end; // ...if
end;

function TGtroDBTreeView.GetLabelFieldName: string;
{*******************************************************************************
Function    : TGtroDBTreeView.GetTitleFieldName
Visibility  : private
Description : Read method for the property TitleFieldName
Revision    : November 15, 2000
*******************************************************************************}
begin
  Result:= FDatalink.FieldNames[3];
end;

procedure TGtroDBTreeView.LoadFromTable;
{*******************************************************************************
Procedure   : TGtroDBTreeView.LoadFromTable
Visibility  : private
Description : Loads the TreeView from the dataset
            : Called by TGtroDBTreeView.ActiveChange
Revision    : 6 December 2000
*******************************************************************************}
var
	R, P, i: Integer;
  T: string;
  F: Boolean;
  Ptr: TItem;
  Item: TItem;
begin
	FListOfItems:= TListOfItems.Create;
  OnChange:= nil;  // no reaction to OnChange event
  Items.BeginUpdate;
  try
    Items.Clear;
    AddRootNode;
    FDatalink.DataSource.DataSet.Open; // opens the dataset
    FDatalink.DataSource.DataSet.First; // go to first item of table
    while not FDatalink.DataSource.DataSet.Eof do
    begin
      R:= FDatalink.Fields[0].AsInteger; // Key field
      P:= FDatalink.Fields[1].AsInteger; // Parent field
      F:= FDatalink.Fields[2].AsBoolean; // IsFolder field
      T:= FDatalink.Fields[3].AsString;  // Title field
      Item:= TItem.Create(F, R, P, T);
      FListOfItems.Add(Item);
      FDatalink.DataSource.DataSet.Next;
    end; // ...for
    { Mise  jour des pointeurs parents }
    for i:= 0 to FListOfItems.Count - 1 do
    begin
      P:= TItem(FListOfItems.Items[i]).ParentID;
      if P <> 0 then
      begin
        Ptr:= FListOfItems.FindItem(P);
        TItem(FListOfItems.Items[i]).ParentItem:= Ptr;
      end; // ...if
    end; // ...for 
    // Construction of the TreeView
    for i:= 0 to FListOfItems.Count - 1 do
      AddANode(TItem(FListOfItems.Items[i]));
    CustomSort(@MyCustomSortProc, 0);
    FullCollapse;
    Items.Item[0].Expand(False);
  finally
    OnChange:= TVChange; // reassign event handler
    FListOfItems.Free; // free List of Items
  	Items.EndUpdate;
  end; // try...finally
end;

procedure TGtroDBTreeView.SetDataSource(Value: TDataSource);
{*******************************************************************************
Procedure   : TGtroDBTreeView.SetDataSource
Visibility  : private
Description : Write method for the DataSource property.
Revision    : 4 December 2000
*******************************************************************************}
begin
  if not (FDataLink.DataSourceFixed and (csLoading in ComponentState)) then
    FDataLink.DataSource:= Value;
  if Value <> nil then Value.FreeNotification(Self);
end;

procedure TGtroDBTreeView.SetKeyFieldName(const Value: string);
{*******************************************************************************
Procedure   : TGtroDBTreeView.SetKeyFieldName
Visibility  : private
Description : Write method for the KeyFieldName property.
Revision    : 4 December 2000
*******************************************************************************}
begin
  FDatalink.FieldNames[0]:= Value;
end;

procedure TGtroDBTreeView.SetIsFolderFieldName(const Value: string);
{*******************************************************************************
Procedure   : TGtroDBTreeView.SetIsFolderFieldName
Visibility  : private
Description : Write method for the IsFolderFieldName property.
Revision    : 4 December 2000
*******************************************************************************}
begin
  FDatalink.FieldNames[2]:= Value;
end;

procedure TGtroDBTreeView.SetParentFieldName(const Value: string);
{*******************************************************************************
Procedure   : TGtroDBTreeView.SetParentFieldName
Visibility  : private
Description : Write method for the ParentFieldName property.
Revision    : 4 December 2000
*******************************************************************************}
begin
  FDatalink.FieldNames[1]:= Value;
end;

procedure TGtroDBTreeView.SetLabelFieldName(const Value: string);
{*******************************************************************************
Procedure   : TGtroDBTreeView.SetTitleFieldName
Visibility  : private
Description : Write method for the TitleFieldName property.
Revision    : 4 December 2000
*******************************************************************************}
begin
  FDatalink.FieldNames[3]:= Value;
end;

procedure TGtroDBTreeView.TVChange(Sender: TObject; Node: TTreeNode);
{*******************************************************************************
Procedure   : TGtroDBTreeView.TreeViewChange
Visibility  : private
Description : Responds to the OnChange event by activating the record of the
            : dataset corresponding to the selected node, assigning a value
            : to the IsContainer property and triggering the OnChange event.
Revision    : 4 December 2000
*******************************************************************************}
begin
	NodeToRecord(Node); // moves record pointer to proper record
	FIsContainer:= IsNodeAllowed(Node);
  FCanDelete:= (not FIsContainer) or (not Node.HasChildren); // 1st May 2005

  if FIsContainer then
    FNavigator.VisibleButtons:= [nbInsert, nbDelete]
  else
    FNavigator.VisibleButtons:= [nbDelete];
  if not FCanDelete then
    FNavigator.VisibleButtons:= FNavigator.VisibleButtons - [nbDelete];

  // Calls the event handler for the OnChange event
  if Assigned(FOnChange) then FOnChange(Sender, Node);
  Node.Selected:= True;
end;

procedure TGtroDBTreeView.TVDragDrop(Sender, Source: TObject; X, Y: Integer);
{*******************************************************************************
Procedure   : TGtroDBTreeView.TVDragDrop
Visibility  : private
Description : Handles the Drop portion of the Drag/Drop operation
Revision    : 4 December 2000
*******************************************************************************}
var
	TargetNode, SourceNode: TTreeNode;
  RecordID: Integer;
  Found: Boolean;
begin
  inherited;
	TargetNode:= GetNodeAt(X, Y);
  if TargetNode <> nil then
  begin
    FDataLink.DataSource.DataSet.DisableControls;
    SourceNode:= Selected;
    try
      SourceNode.MoveTo(TargetNode, naAddChildFirst);
      TargetNode.Expand(False);
      Selected:= TargetNode;
      { dition de la table }
      // Trouver record correspondant au noeud source et le rendre courant
      Found:= NodeToRecord(TargetNode);
      if Found then
        RecordID:= FDataLink.Fields[0].Value
      else RecordID:= 0;
      // Trouver record correspondant au noeud cible et le rendre courant
      NodeToRecord(SourceNode);
      FDataLink.DataSource.DataSet.Edit;
      FDataLink.Fields[1].Value:= RecordID;  // change le record Parent
      FDataLink.DataSource.DataSet.Post;
      CustomSort(@MyCustomSortProc, 0);
    finally
      Selected:= SourceNode;
    	FDataLink.DataSource.DataSet.EnableControls;
    end; // try...finally
  end; // ...if
end;

procedure TGtroDBTreeView.TVDragOver(Sender, Source: TObject; X, Y: Integer;
{*******************************************************************************
Procedure   : TGtroDBTreeView.TVDragOver
Visibility  : private
Description : Handles the DragOver portion of the Drag/Drop operation
Revision    : 4 December 2000
*******************************************************************************}
  State: TDragState; var Accept: Boolean);
const
	EDGE = 50;
var
	TargetNode, SourceNode: TTreeNode;
begin
  inherited;
	TargetNode:= GetNodeAt(X, Y);
  if (Source = Sender) and (TargetNode <> nil) then
  begin
		Accept:= IsNodeAllowed(TargetNode);
  	if Accept then
  	begin
  		SourceNode:= Selected;
  		while (TargetNode.Parent <> nil) and (TargetNode <> SourceNode) do
  			TargetNode:= TargetNode.Parent;
  		if TargetNode = SourceNode then Accept:= False;
  	end; // ...if
  end // ...if
  else Accept:= False;
  // Scrolling
  if (Y < EDGE) and Assigned(TopItem.GetPrevVisible()) then
  	TopItem:= TopItem.GetPrevVisible;
  if (Y > Height - EDGE)
  	and Assigned(TopItem.GetNextVisible()) then
  	TopItem:= TopItem.GetNextVisible;
end;

{*******************************************************************************
                     TGtroDBTreeView - Protected Methods
*******************************************************************************}

procedure TGtroDBTreeView.Edit(const Item: TTVItem);
{*******************************************************************************
Procedure   : TGtroDBTreeView.Edit
Visibility  : protected
Description : Handles the OnEdited event
Revision    : 4 December 2000
*******************************************************************************}
begin
  inherited Edit(Item);
  CustomSort(@MyCustomSortProc, 0);
end;

procedure TGtroDBTreeView.Loaded;
{*******************************************************************************
Procedure   : TGtroDBTreeView.Loaded
Visibility  : protected
Description : Assigns OnChange and OnEdited events of the TreeView
            : to FOnchange and FOnEdited if they were assigned at design time;
            : Assigns TreeViewChange and Edited events to the same events
Revision    : 4 December 2000
*******************************************************************************}
begin
  inherited Loaded;
  if Assigned(OnChange) then FOnChange:= OnChange;
  if Assigned(OnClick) then FOnClick:= OnClick;
  if Assigned(OnEdited) then FOnEdited:= OnEdited;
  if Assigned(OnEnter) then FOnEnter:= OnEnter;
  if Assigned(OnExit) then FOnExit:= OnExit;
  OnChange:= TVChange;
  OnClick:= TVClick;
  OnEdited:= Edited;
  OnEnter:= TVEnter;
  OnExit:= TVExit;
end;

function TGtroDBTreeView.NodeToRecord(Node: TTreeNode): Boolean;
{*******************************************************************************
Function    : TGtroDBTreeView.NodeToRecord
Visibility  : protected
Description : Given a node, locates the corresponding field of the dataset
            : and actives it.
            : Called by TGtroDBTreeView.TreeViewChange,
            : TGtroDBTreeView.DeleteSelectedNode, TGtroDBTreeView.TVDragDrop
Revision    : 6 December 2000
*******************************************************************************}
begin
  FDataLink.DataSet.DisableControls;
  try
    Result:= FDataLink.DataSource.DataSet.Locate(FDatalink.FFieldNames[0],
      Integer(Node.Data), []);
  finally
    FDataLink.DataSet.EnableControls;
  end; // try...finally
end;

{*******************************************************************************
                     TGtroDBTreeView - Public Methods
*******************************************************************************}

constructor TGtroDBTreeView.Create(AOwner: TComponent);
{*******************************************************************************
Destructor  : TGtroDBTreeView.Create
Visibility  : public
Description : Initialization of the object
Revision    : 25 November 2000, 13 Octobwer 2007
*******************************************************************************}
var
  Sfi: TSHFileInfo;
  FilePath: string;
  i: Integer;
  Icon: TIcon;
begin
  inherited Create(AOwner);
  FDataLink:= TTreeViewDataLink.Create(Self);
  FDataLink.TreeView:= Self;
  FDataLink.OnActiveChange:= ActiveChange;
  DragMode:= dmAutomatic;
  OnDragOver:= TVDragOver;
  OnDragDrop:= TVDragDrop;
  if not Assigned(Images) then // 14 Oct 07
  begin
    FilePath:= 'C:\';
    FImageList:= TImageList.Create(Self);
    FImageList.Masked:= False;
    // Fetch the system icons
    FImageList.Handle:= SHGetFileInfo(PChar(FilePath), 0, sfi, SizeOf(TSHFileInfo),
      SHGFI_SYSICONINDEX or SHGFI_SMALLICON);
    FImageList.ShareImages := TRUE;
    // Rearrange them
    FImageList.Move(2, 0);
    FImageList.Move(2, 1);
    for i:= FImageList.Count - 1 downto 3 do
      FImageList.Delete(i);
    Icon:= TIcon.Create;
    try
      FImageList.GetIcon(1, Icon);
      FImageList.AddIcon(Icon);
      FImageList.GetIcon(2, Icon);
      FimageList.AddIcon(Icon);
    finally
      Icon.Free;
    end; // try...finally
    Images:= FImageList;
  end;
end;

destructor TGtroDBTreeView.Destroy;
{*******************************************************************************
Destructor  : TGtroDBTreeView.Destroy
Visibility  : public
Description : Cleanup code
Revision    : 4 December 2000
*******************************************************************************}
begin
  FDataLink.Free;
  FDataLink:= nil;
  inherited Destroy;
end;

procedure TGtroDBTreeView.AddNewNode(NodeType: eNodeType);
{*******************************************************************************
Procedure   : TGtroDBTreeView.AddNewNode
Visibility  : public
Description : Adds a new node of the eNodeType and creates a new record in the
            : dataset
Revision    : 6 December 2000
*******************************************************************************}
var
  RecordID, ParentID: Integer;
  NewNode: TTreeNode;
begin
	if Selected = nil then Selected:= Items.GetFirstNode;
  //if not IsNodeAllowed(Selected) then Exit;
  RecordID:= NextRecordID;
  ParentID:= Integer(Selected.Data);  // RefNo du noeud parent
  NewNode:= Items.AddChildObjectFirst(Selected,
  	'     ', Pointer(RecordID));
    with NewNode do
    begin
    	case NodeType of
        ntFolder:
        	begin
    				ImageIndex:= IMG_FOLDER_CLOSED;
      			SelectedIndex:= IMG_FOLDER_OPEN;
          end;
        ntFile:
        	begin
    				ImageIndex:= IMG_FILE_CLOSED;
      			SelectedIndex:= IMG_FILE_OPEN;
          end;
      end; // ...case
      MakeVisible;
    end; // ...with
    Selected:= NewNode;
    NewNode.Focused:= true;
    NewNode.EditText;
  { Inscription dans la table }
  with FDataLink.DataSource.DataSet do
  begin
  	Append;
  	FDatalink.Fields[0].Value:= RecordID;
  	FDatalink.Fields[1].Value:= ParentID;
  	FDatalink.Fields[2].Value:= NodeType = ntFolder;
  	FDatalink.Fields[3].Value:= NewNode.Text;
  	Post;
    GetNextRecordID;
  end; // ...with
end;

procedure TGtroDBTreeView.DeleteSelectedNode; // public
{*******************************************************************************
Procedure   : TGtroDBTreeView.DeleteSelectedNode
Visibility  : public
Description : Deletes the selected node
Revision    : 6 December 2000; 1 May 2005
*******************************************************************************}
begin
  NodeToRecord(Selected);
  Selected.Delete;
end;

procedure TGtroDBTreeView.Focus;
begin
  Selected.Focused:= true;
  TVChange(Self, Selected);
end;

{*******************************************************************************
                            Global procedures
*******************************************************************************}

procedure Register;
begin
  RegisterComponents('GTRO', [TGtroDBTreeView]);
end;

function TGtroDBTreeView.SelectNode(RecID: Integer): Boolean;
var
  Node: TTreeNode;
begin
  Result:= False;
  Node:= Selected;
  if (Integer(Node.Data)) <> RecId then
  begin
    FullCollapse;
    Node:= Items.GetFirstNode;
    while (Node <> nil) do
    begin
      if RecID = Integer(Node.Data) then
      begin
        Result:= true;
        Node.Expand(False);
        Node.Selected:= True;
        Exit;
      end; // if RecID
      Node:= Node.GetNext;
    end; // while
  end; // if Node.Data
end;

procedure TGtroDBTreeView.NavigatorClick(Sender: TObject;
  Button: TNavigateBtn);
begin
  // Excute le gestionnaire d'vnement extrieur associ au navigateur
  if Assigned(FOldOnClick) then FOldOnClick(Self, Button);
  // Supprime le noeud slectionn de la vue hirarchique
  if Button = nbDelete then DeleteSelectedNode;
  // Ajoute un noeuf enfant au noeud slectionn
  if Button = nbInsert then
  begin
    // Le type de ce noeud est spcifi par la proprit NodeType
    NodeType:= ntFile;
    if MessageDlg('Noeud rpertoire?', mtConfirmation, [mbYes, mbNo], 0) = mrYes then
      NodeType:= ntFolder;
    AddNewNode(NodeType);
  end;
end;

procedure TGtroDBTreeView.ConnectNavigator(Navigator: TDBNavigator);
begin
  if Assigned(FExtNavigator) then
  begin
    // Mmorise le gestionnaire d'vnement associ au navigateur externe
    // afin de l'appeler lors du dclanchement de OnClick
    FNavigator:= FExtNavigator;
    FOldOnClick:= FExtNavigator.OnClick;
    RemButtons:= FNavigator.VisibleButtons;
    FNavigator.VisibleButtons:= [nbInsert, nbDelete]; // deux boutons
    FNavigator.DataSource:= DataSource; // assigne la source de donnes
    FNavigator.OnClick:= NavigatorClick; // assigne le gestionnaire d'vnement interne
    OnChange:= TVChange;
    OnEdited:= Edited;
    TVChange(Self, Selected);
  end;
end;

procedure TGtroDBTreeView.DisconnectNavigator;
begin
  if Assigned(FNavigator) then  //  moins que FNavigator ne soit dj nil.
  begin
    FNavigator.VisibleButtons:= RemButtons;
    FNavigator.OnClick:= nil;
    OnChange:= nil;
    OnEdited:= nil;
    FNavigator.DataSource:= nil;
    FNavigator:= nil;
  end;
end;

procedure TGtroDBTreeView.TVEnter(Sender: TObject);
begin

  ConnectNavigator(FExtNavigator);
  if Assigned(FOnEnter) then FOnEnter(Self);
end;

procedure TGtroDBTreeView.TVExit(Sender: TObject);
begin
  DisconnectNavigator;
  if Assigned(FOnExit) then FOnExit(Self);
end;

procedure TGtroDBTreeView.TVClick(Sender: TObject);
begin
  if Assigned(FOnClick) then FOnclick(Self);
  
end;

end.
